package be.ac.ulb.mlg.utils;

/*
 * The MIT License (MIT)
 * 
 * Copyright (c) 2013 Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * 
 * @author Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * @version 1.00, 24/06/2013
 */

public interface Renormalizer{
	/**
	 * Apply the normalization process on the given vector of data vectors
	 * @param input The given vector of data vectors
	 * @return the input array
	 */
	public double[][] normalizeInput(final double[][] input);
	/**
	 * Apply the normalization process on the given vector of data vectors and result measures
	 * @param input The given vector of data vectors
	 * @param output The result vector (measures of data vectors)
	 * @param measure The corresponding measure
	 * @return the output array
	 */
	public double[][] normalizeOutput(final double[][] input,final double[][] output,final Measure measure);
	/**
	 * Know if the current renormalizer owns a native implementation
	 * @return true if the native implementation exists
	 */
	public boolean hasNativeImplementation();
	/**
	 * Know if the renormalizer can handle input (call normalizeInput has effect)
	 * @return true if it can handle
	 */
	public boolean processInput();
	/**
	 * Know if the renormalizer can handle output (call normalizeOutput has effect)
	 * @return true if it can handle
	 */
	public boolean processOutput();
	/**
	 * Know if the measurer use pre-shuffle
	 * @return true if pair will be shuffled, false otherwise
	 */
	public boolean shufflePair();
	/**
	 * Method called when pairwise renormalization is used in order to obtain a normalized version of the shuffled vector as if it was in the original input matrix
	 * @param shuffledFirst The first vector of the pairwise comparison (shuffled version)
	 * @param shuffledSecond The second vector of the pairwise comparison (shuffled version)
	 * @param first The index of the first vector from the original input matrix
	 * @param second The index of the second vector from the original input matrix
	 */
	public void simulateRenormalization(double[] shuffledFirst,double[] shuffledSecond,final int first,final int second);
	/**
	 * Method called when pairwise renormalization is used in order to ignore or not a row from the original input matrix
	 * @param vect A vector which is considerate as a row from the original input matrix corresponding to the row at the given index
	 * @param index The index of the vector to map it into the original matrix at this index
	 * @param ignore A boolean to tell if the values of the vector must be ignored (typically remove these values from the normalization process as if it was the row "index" from the original input matrix) with the true value (typically false to add it into renormalization)
	 */
	public void updateRenormalization(double[] vect,int index,boolean ignore);//TODO
}
