package be.ac.ulb.mlg.utils;

/*
 * The MIT License (MIT)
 * 
 * Copyright (c) 2013 Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * 
 * @author Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * @version 1.00, 24/06/2013
 */

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.DoubleBuffer;
import java.nio.IntBuffer;
import java.util.Arrays;

/**
 * 
 * @author Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * @version 1.00, 24/06/2013
 */

/**
 * This class contains usefull methods to pass raw data to a native implementation through ByteBuffer 
 *
 */
final class JNIUtils {

	/**
	 * Create a direct ByteBuffer according to the native order
	 * @param count The number of byte
	 * @return The direct ByteBuffer
	 */
	static final ByteBuffer createByteBuffer(final int count){
		final ByteBuffer debug = ByteBuffer.allocateDirect(count).order(ByteOrder.nativeOrder());
		if(!debug.isDirect()) throw new OutOfMemoryError("Cannot allocate native memory block");	
		return debug;
	}
	/**
	 * Create a direct DoubleBuffer (from ByteBuffer) according to the native order
	 * @param count The number of byte
	 * @return The direct ByteBuffer
	 */
	static final DoubleBuffer createDoubleBuffer(final int count){
		return createByteBuffer(count*Double.SIZE/Byte.SIZE).asDoubleBuffer();
	}
	/**
	 * Create a direct IntBuffer (from ByteBuffer) according to the native order
	 * @param count The number of byte
	 * @return The direct ByteBuffer
	 */
	static final IntBuffer createIntBuffer(final int count){
		return createByteBuffer(count*Integer.SIZE/Byte.SIZE).asIntBuffer();
	}
	/**
	 * Make a copy of a raw array 
	 * @param buff The array to copy
	 * @return The copy as a raw double array
	 */
	public static double[][] toArray(double[][] buff) {
		final double[][] copy = new double[buff.length][];
		for(int i=0;i<copy.length;i++)
			copy[i] = Arrays.copyOf(buff[i],buff[i].length);
		return copy;
	}
	/**
	 * Make a copy of a DoubleBuffer
	 * @param buff The DoubleBuffer to copy
	 * @param  columnCount The size of columns
	 * @return The copy as a raw double array
	 */
	public static double[][] toArray(DoubleBuffer[] buff,int columnCount) {
		final double[][] copy = new double[buff.length][columnCount];
		for(int i=0;i<copy.length;i++)
			buff[i].get(copy[i]);
		return copy;
	}
}
