package be.ac.ulb.mlg.utils.measure;

/*
 * The MIT License (MIT)
 * 
 * Copyright (c) 2013 Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * 
 * @author Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * @version 1.00, 30/08/2013
 */

import be.ac.ulb.mlg.utils.Measure;
import be.ac.ulb.mlg.utils.MeasureUtils;

/**
 * Abstract class of entropy that need to use an estimator.
 * The measure function of the {@link Measure} interface computes the joint entropy.
 */
public abstract class Entropy implements Measure{
	
	/**
	 * Compute the entropy of a vector of values (non missing values (NaN))
	 * @param frequencies The occurrences array (of values)
	 * @param distinctValues The number of distinct values
	 * @param numberOfValues The number of values
	 * @return The pseudo count
	 */
	public abstract double entropy(final double[] frequencies,final int distinctValues,final int numberOfValues);

	/**
	 * Compute the joint entropy of two vectors of values x,y (non missing values (NaN))
	 * @param x The first vector of values
	 * @param y The second vector of values
	 * @return The joint entropy estimation
	 */
	public double jointEntropy(double[] x,double[] y) {
		final double[] frequencies = new double[x.length];
		final long informations = MeasureUtils.countOccurencies(new double[][]{y,x}, frequencies, null);
		return entropy(frequencies,((int)(informations>>32)),((int)(informations)));
	}
	
	/**
	 * Compute the entropy of a vector of values (non missing values (NaN))
	 * @param x The vector of values
	 * @return The entropy estimation
	 */
	public double entropy(final double[] x) {
		final double[] frequencies = new double[x.length];
		final long informations = MeasureUtils.countOccurencies(x, frequencies,false);
		return entropy(frequencies,((int)(informations>>32)),((int)(informations)));
	}
	
	@Override
	public double measure(double[] a, double[] b, boolean[] computable) {
		long informations;
		final double[] frequencies = new double[a.length];
		informations = MeasureUtils.countOccurencies(new double[][]{b,a}, frequencies, computable);
		return entropy(frequencies,((int)(informations>>32)),((int)(informations)));
	}

	@Override
	public boolean hasNativeImplementation() {
		return false;
	}

	@Override
	public boolean requireDataTransformation() {
		return false;
	}

	@Override
	public void transform(double[][] data) {}
}
