package be.ac.ulb.mlg.utils.measure;

/*
 * The MIT License (MIT)
 * 
 * Copyright (c) 2013 Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * 
 * @author Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * @version 1.00, 24/06/2013
 * @see"Theory of linear operators in Hilbert space (N.I. Akhiezer, I.M. Glazman)"
 */

import be.ac.ulb.mlg.utils.Measure;
import be.ac.ulb.mlg.utils.MeasureUtils;
import be.ac.ulb.mlg.utils.measure.kernel.GaussianKernel;

/**
 * Compute the Hilbert-Schmidt independence criterion according to
 * the estimate in the paper "On Kernel Parameter Selection in Hilbert-Schmidt Independence Criterion" p.3	
 */
public class HilbertSchmidt implements Measure{
	private final Kernel kernelA,kernelB;
	
	/**
	 * Initializing the Hilbert-Schmidt independence criterion with specified kernels
	 * @param kernelA kernel for the first vector values
	 * @param kernelA kernel for the second vector values
	 */
	public HilbertSchmidt(final Kernel kernelA,final Kernel kernelB){
		this.kernelA = kernelA;
		this.kernelB = kernelB;
	}
	/**
	 * Initializing the Hilbert-Schmidt independence criterion to use the Gaussian Kernel
	 */
	public HilbertSchmidt(){
		this(new GaussianKernel(),new GaussianKernel());
	}
	
	@Override
	public double measure(final double[] A, final double[] B,boolean[] computable){
		kernelA.setup(A,computable != null);
		kernelB.setup(B,computable != null);
		
		int n = A.length;
		int i,j,k,l,m;
		double trace = 0.0;
		double tmp1,tmp2,tmp3;//used to "prevent" error during accumulations (large + small)
		
		if(computable != null) {
			for(i=0;i<computable.length;i++)
				if(!computable[i])
					--n;
		}
		
		final double gamma0 = -1.0 / n;
		final double gamma1 = 1.0 + gamma0;
		
		if(computable != null) {
			int[] notNaN = new int[n];
			for(i=0,j=0;i<n;i++,j++){
				while(!computable[j]) ++j;
				notNaN[i]=j;
			}
			for(i=0;i<n;i++){
				j = i;
				tmp3 = 0.0;
				for(m=0;m<n;m++){
					tmp2 = 0.0;
					for(l=0;l<n;l++){
						tmp1 = 0.0;
						for(k=0;k<n;k++){
							tmp1 += (k!=l?gamma0:gamma1) * kernelA.ker(A[notNaN[i]],A[notNaN[k]]);
						}
						tmp2 += tmp1 * kernelB.ker(B[notNaN[l]],B[notNaN[m]]);
					}
					tmp3 += tmp2 * (m!=i?gamma0:gamma1);
				}
				trace += tmp3;
			}
		}else{
			for(i=0;i<n;i++){
				j = i;
				tmp3 = 0.0;
				for(m=0;m<n;m++){
					tmp2 = 0.0;
					for(l=0;l<n;l++){
						tmp1 = 0.0;
						for(k=0;k<n;k++){
							tmp1 += (k!=l?gamma0:gamma1) * kernelA.ker(A[i],A[k]);
						}
						tmp2 += tmp1 * kernelB.ker(B[l],B[m]);
					}
					tmp3 += tmp2 * (m!=i?gamma0:gamma1);
				}
				trace += tmp3;
			}
		}
		return trace/(n*n);
	}
	
	public static double norm(final double[] A, final double[] B,boolean[] computable){
		double frobenius = 0.0;
		int index;
		
		// no need of the absolute value because Hilbert Schmidt uses power 2 => x*x is always +
		if(computable != null){
			for(index=0;index<A.length;index++)
				if(computable[index])
					frobenius += A[index]*A[index] + B[index]*B[index];
		}else{
			for(index=0;index<A.length;index++)
				frobenius += A[index]*A[index] + B[index]*B[index];
		}
		
		return MeasureUtils.sqrt(frobenius);
	}
	
	@Override
	public boolean hasNativeImplementation() {
		return	kernelA.getClass().equals(GaussianKernel.class)
			&&	kernelB.getClass().equals(GaussianKernel.class);
	}
	@Override
	public boolean requireDataTransformation() {
		return false;
	}

	@Override
	public void transform(double[][] data) {
		//nothnig to do
	}
}
