package be.ac.ulb.mlg.utils.measure;

/*
 * The MIT License (MIT)
 * 
 * Copyright (c) 2013 Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * 
 * @author Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * @version 1.00, 24/06/2013
 * @see <a href="http://www.amazon.com/Handbook-Research-Synthesis-Meta-Analysis/dp/0871541637">The Handbook of Research Synthesis and Meta-analysis p.364 (Harris M. Cooper, Larry V. Hedges, Jeff C. Valentine)</a>
 */

import static java.lang.Math.exp;
import static java.lang.Math.log;
import be.ac.ulb.mlg.utils.Measure;
import be.ac.ulb.mlg.utils.MeasureUtils;

/**
Variance of log ratios scaled to [0;1]: 1-exp(-sqrt(D(x,y))) according to Aitchison
where D(X,Y) is the Variance of log ratios.
*/
public class VarianceOfLogRatios implements Measure{
	@Override
	public double measure(final double[] A, final double[] B,boolean[] computable){
		double mean		= 0.0;
		double square	= 0.0;
		double logRatio;
		int index;
		int count = A.length;
		
		if(computable != null){
			for(index=0;index<A.length;index++)
				if(computable[index]){
					logRatio = log(A[index]/B[index]);
					mean	+= logRatio;
					square	+= logRatio*logRatio;
				}else count--;
		}else{
			for(index=0;index<A.length;index++){
				logRatio = log(A[index]/B[index]);
				mean	+= logRatio;
				square	+= logRatio*logRatio;
			}
		}
		
		if(count < 1)return ERROR_VALUE;
		
		mean	/=count;
		square	/=count;
		
		return 1.0 - exp(-MeasureUtils.sqrt(square - mean*mean));
	}
	@Override
	public boolean hasNativeImplementation() {
		return true;
	}
	@Override
	public boolean requireDataTransformation() {
		return false;
	}

	@Override
	public void transform(double[][] data) {
		//nothnig to do
	}
}
