package be.ac.ulb.mlg.utils.measure.entropy;

/*
 * The MIT License (MIT)
 * 
 * Copyright (c) 2013 Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * 
 * @author Jean-Sebastien Lerat (Jean-Sebastien.Lerat@ulb.ac.be)
 * @version 1.00, 30/08/2013
 */

import static java.lang.Math.log;
import be.ac.ulb.mlg.utils.measure.Entropy;

/**
 * Abstract Dirichlet probability distribution for entropy estimator.
 */
public abstract class DirichletEntropy extends Entropy{
	/**
	 * The Euler-Mascheroni constant
	 */
	public static final double EULER_MASCHERONI = 0.57721566490153286060651209008240243104215933593992;
	
	/**
	 * Model estimator for a dirichlet probability
	 * @param frequencies The occurrences array (of values)
	 * @param bins The number of bins (number of distinct values)
	 * @param samples The number of samples (number of values)
	 * @return The pseudo count
	 */
	protected abstract double estimateModel(final double[] frequencies,final int bins, final int samples);
	
	@Override
	public double entropy(final double[] frequencies, final int distinctValues, final int numberOfValues){
		double sum = 0.0;
		
		final double pseudoCount		= estimateModel(frequencies,distinctValues,numberOfValues);
		final double binsXpseudoCount	= distinctValues * pseudoCount;
		
		for(double ai:frequencies){
			if(ai>0.0)
				sum += (ai + pseudoCount) * (digamma( numberOfValues + binsXpseudoCount + 1.0) - digamma(ai + pseudoCount + 1.0));
		}
		
		return sum / (numberOfValues + binsXpseudoCount);
  	}
	/**
	 * Digamma function inspired from commons-apache
	 * @param x the value ( > 0) for which digamma will be estimated
	 * @return the estimation of digamma(x)
	 */
	private static double digamma(double x) {
		if (x <= 1E-5) 
			return -EULER_MASCHERONI - 1.0 / x;
		
		double acc = 0.0;
		while(x < 49){
			acc += 1.0 / x;
			x	+= 1.0;
		}
		
		final double inv = 1.0 / (x * x);
		return log(x) - 0.5 / x - inv * ((1.0 / 12.0) + inv * (1.0 / 120.0 - inv / 252.0)) - acc;
	}
	
	@Override
	public boolean hasNativeImplementation() {
		return false;
	}

	@Override
	public boolean requireDataTransformation() {
		return false;
	}

	@Override
	public void transform(double[][] data) {
	}
}
